<?php

require_once 'lib/opcodes.php';
require_once 'lib/types.php';

class Machine {
    private Components $components;
    private Processor $processor;
    private array $interruptHandlers = [];

    public function __construct() {
        $this->components = new Components($depth = 4);
        $this->processor = new Processor($depth, $this->components);
    }

    public function run() : void {
        while(true) {
            $this->processor->tick();
            foreach($this->interruptHandlers as &$handler) {
                $handler();
            }
        }
    }
    public function attachInterrupt(Closure $handler) : void {
        $this->interruptHandlers[] = $handler;
    }
}

class Components {
    private array $components;

    public function __construct(int $depth) {
        $this->components = (new LoadComponents("components.txt", $depth))->getComponents();
    }

    public function read(int $byte) : int {
        foreach($this->components as &$component) {
            if($component->isByteBelongToComponent($byte)) {
                return $component->component->read($byte - $component->getOffset());
            }
        }
        return 0;
    }
    public function write(int $byte, int $value) : void {
        foreach($this->components as &$component) {
            if($component->isByteBelongToComponent($byte)) {
                $component->component->write($byte - $component->getOffset(), $value);
                return;
            }
        }
    }
}

class LoadComponents {
    private array $components;
    private array $endComponents;
    private string $filename;
    private int $depth;

    private function loadComponent(string $name, string $argv, bool $endLoad = false) : void {
        try {
            require_once "components/$name.php";
            $component = new $name($argv);
            if($endLoad) {
                $this->endComponents[] = $component;
            } else {
                $this->components[] = $component;
            }
        } catch(Throwable $e) {
            echo "[ERROR] Can't load $name: " . $e->getMessage() . PHP_EOL;
            die();
        }
    }
    private function loadFromFile(string $filename) : void {
        $components = explode("\n", file_get_contents($filename));
        $i = 0;
        $endLoad = false;
        while(true) {
            $componentString = explode(' ', $components[$i], 2);
            if(count($componentString) === 1) $componentString[1] = '';
            $this->loadComponent($componentString[0], $componentString[1], $endLoad);
            $i++;
            if($i > count($components) - 1) break;
            if($components[$i] === 'empty') {
                $endLoad = true;
                $i++;
            }
        }
    }

    public function __construct(string $filename, int $depth) {
        $this->components = [];
        $this->endComponents = [];
        $this->filename = $filename;
        $this->depth = $depth;
    }
    
    public function getComponents() : array {
        $components = [];
        $this->loadFromFile($this->filename);
        $offset = 0;
        foreach($this->components as &$component) {
            $components[] = new LoadedComponent($offset, $component);
            $offset += $component->bytes;
        }
        $offset = 2 ** ($this->depth * 8) - 1;
        $endComponents = array_reverse($this->endComponents);
        foreach($endComponents as &$component) {
            $components[] = new LoadedComponent($offset, $component);
            $offset -= $component->bytes;
        }
        return $components;
    }
}

class LoadedComponent {
    private int $offset;
    private int $bytes;
    public readonly Component $component;
    
    public function __construct(int $offset, Component $component) {
        $this->offset = $offset;
        $this->bytes = $component->bytes;
        $this->component = $component;
    }
    
    public function getOffset() : int {
        return $this->offset;
    }
    public function getBytes() : int {
        return $this->bytes;
    }

    public function isByteBelongToComponent(int $byte) : bool {
        return $byte >= $this->getOffset() & $byte < $this->getOffset() + $this->getBytes();
    }
}

class Processor {
    private readonly int $depth;
    private Components $components;
    private Opcodes $opcodes;
    private Register $programCounter;
    private Registers $registers;

    public function __construct(int $depth, Components &$components) {
        $this->depth = $depth;
        $this->components = $components;
        $this->registers = new Registers($depth);
        $this->programCounter = new Register($depth);
        $this->opcodes = new Opcodes($this->components, $this->registers, $depth, $this->programCounter);
    }

    public function tick() : void {
        $this->opcodes->execute($this->components->read($this->programCounter->read()));
        $this->programCounter->write($this->programCounter->read() + 1);
    }
}

class Registers {
    private array $registers;

    public function __construct(int &$depth) {
        $registers = [];
        for($i = 1; $i <= 8; $i++) {
            $registers[] = new Register($depth);
        }
        $this->registers = $registers;
    }

    public function read(int $register) : int {
        return $this->registers[$register]->read();
    }
    public function write(int $register, int $value) : void {
        $this->registers[$register]->write($value);
    }
}

class Register {
    private int $value;
    private int $depth;

    public function __construct(&$depth) {
        $this->value = 0;
        $this->depth = &$depth;
    }

    public function read() : int {
        return $this->value;
    }
    public function write(int $value) : void {
        $this->value = $value & intval((2 ** ($this->depth * 8)) - 1);
    }
}

class Opcodes {
    private array $opcodes;

    public function __construct(
        Components &$components,
        Registers &$registers,
        int &$depth,
        Register &$programCounter
    ) {
        $opcodes = [];

        $opcodes[] = new MOV($components, $registers, $depth, $programCounter);
        $opcodes[] = new NAND($components, $registers, $depth, $programCounter);
        $opcodes[] = new BIT($components, $registers, $depth, $programCounter);
        $opcodes[] = new JMPC($components, $registers, $depth, $programCounter);

        $this->opcodes = $opcodes;
    }

    public function execute(int $byte) : void {
        $this->opcodes[$byte >> 6]->execute($byte & 0b111111);
    }
}

class busrouter extends Component {
    private string $contents;
    public readonly int $bytes;

    public function __construct(array $components) {
        foreach($components as &$component) {
            
        }
    }

    public function read(int $byte) : int {
        return 0;
    }
    public function write(int $byte, int $value) : void {

    }
}



$machine = new Machine;
// $machine->attachInterrupt(
//     function() {
//         readline();
//     }
// );
$machine->run();